import os
import sqlite3
import json
import time
from typing import Any, Optional, Dict, List, Tuple

def _now() -> int:
    return int(time.time())

class DB:
    def __init__(self, db_path: str):
        os.makedirs(os.path.dirname(db_path), exist_ok=True)
        self.db_path = db_path
        self._init()

    def _conn(self) -> sqlite3.Connection:
        conn = sqlite3.connect(self.db_path, check_same_thread=False)
        conn.row_factory = sqlite3.Row
        return conn

    def _init(self) -> None:
        with self._conn() as c:
            c.executescript("""
            CREATE TABLE IF NOT EXISTS settings (
                key TEXT PRIMARY KEY,
                value TEXT NOT NULL
            );

            CREATE TABLE IF NOT EXISTS users (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                tg_id INTEGER UNIQUE NOT NULL,
                username TEXT,
                phone TEXT,
                profile_text TEXT,
                state TEXT,
                tmp_json TEXT,
                is_verified INTEGER NOT NULL DEFAULT 0,
                is_active INTEGER NOT NULL DEFAULT 1,
                msg_count INTEGER NOT NULL DEFAULT 0,
                last_seen INTEGER NOT NULL DEFAULT 0,
                created_at INTEGER NOT NULL
            );

            CREATE TABLE IF NOT EXISTS admins (
                tg_id INTEGER PRIMARY KEY,
                role TEXT NOT NULL DEFAULT 'admin',
                added_at INTEGER NOT NULL
            );

            CREATE TABLE IF NOT EXISTS courses (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT NOT NULL,
                description TEXT,
                is_active INTEGER NOT NULL DEFAULT 1,
                created_at INTEGER NOT NULL
            );

            CREATE TABLE IF NOT EXISTS lessons (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                course_id INTEGER NOT NULL,
                title TEXT NOT NULL,
                status TEXT NOT NULL DEFAULT 'processing',
                created_at INTEGER NOT NULL,
                FOREIGN KEY(course_id) REFERENCES courses(id)
            );

            CREATE TABLE IF NOT EXISTS media (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                lesson_id INTEGER NOT NULL,
                quality TEXT NOT NULL,     -- e.g. 360p / 480p / 720p
                path TEXT NOT NULL,
                status TEXT NOT NULL DEFAULT 'ready',
                size_bytes INTEGER NOT NULL DEFAULT 0,
                created_at INTEGER NOT NULL,
                FOREIGN KEY(lesson_id) REFERENCES lessons(id)
            );

            CREATE TABLE IF NOT EXISTS uploads (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                course_id INTEGER NOT NULL,
                lesson_id INTEGER NOT NULL,
                tg_file_id TEXT NOT NULL,
                original_path TEXT NOT NULL,
                status TEXT NOT NULL DEFAULT 'pending', -- pending|processing|done|error
                error TEXT,
                created_at INTEGER NOT NULL,
                FOREIGN KEY(course_id) REFERENCES courses(id),
                FOREIGN KEY(lesson_id) REFERENCES lessons(id)
            );

            CREATE TABLE IF NOT EXISTS tokens (
                token TEXT PRIMARY KEY,
                tg_id INTEGER NOT NULL,
                media_id INTEGER NOT NULL,
                expires_at INTEGER NOT NULL,
                created_at INTEGER NOT NULL,
                FOREIGN KEY(media_id) REFERENCES media(id)
            );
            """)

    # Settings
    def get_setting(self, key: str, default: Optional[str] = None) -> Optional[str]:
        with self._conn() as c:
            row = c.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
            return row["value"] if row else default

    def set_setting(self, key: str, value: str) -> None:
        with self._conn() as c:
            c.execute(
                "INSERT INTO settings(key,value) VALUES(?,?) "
                "ON CONFLICT(key) DO UPDATE SET value=excluded.value",
                (key, value),
            )

    # Users
    def upsert_user_seen(self, tg_id: int, username: Optional[str]) -> None:
        now = _now()
        with self._conn() as c:
            row = c.execute("SELECT id FROM users WHERE tg_id=?", (tg_id,)).fetchone()
            if row:
                c.execute(
                    "UPDATE users SET username=?, last_seen=?, msg_count=msg_count+1 WHERE tg_id=?",
                    (username, now, tg_id),
                )
            else:
                c.execute(
                    "INSERT INTO users(tg_id, username, created_at, last_seen, msg_count) VALUES(?,?,?,?,?)",
                    (tg_id, username, now, now, 1),
                )

    def get_user(self, tg_id: int) -> Optional[sqlite3.Row]:
        with self._conn() as c:
            return c.execute("SELECT * FROM users WHERE tg_id=?", (tg_id,)).fetchone()

    def set_user_phone(self, tg_id: int, phone: str) -> None:
        with self._conn() as c:
            c.execute("UPDATE users SET phone=? WHERE tg_id=?", (phone, tg_id))

    def set_user_profile_text(self, tg_id: int, text: str) -> None:
        with self._conn() as c:
            c.execute("UPDATE users SET profile_text=? WHERE tg_id=?", (text, tg_id))

    def set_user_state(self, tg_id: int, state: Optional[str], tmp: Optional[Dict[str, Any]] = None) -> None:
        tmp_json = json.dumps(tmp or {}, ensure_ascii=False)
        with self._conn() as c:
            c.execute("UPDATE users SET state=?, tmp_json=? WHERE tg_id=?", (state, tmp_json, tg_id))

    def verify_user(self, tg_id: int, verified: bool) -> None:
        with self._conn() as c:
            c.execute("UPDATE users SET is_verified=? WHERE tg_id=?", (1 if verified else 0, tg_id))

    def list_users(self) -> List[sqlite3.Row]:
        with self._conn() as c:
            return c.execute("SELECT * FROM users ORDER BY created_at DESC").fetchall()

    # Admins
    def is_admin(self, tg_id: int) -> bool:
        with self._conn() as c:
            row = c.execute("SELECT tg_id FROM admins WHERE tg_id=?", (tg_id,)).fetchone()
            return bool(row)

    def add_admin(self, tg_id: int) -> None:
        with self._conn() as c:
            c.execute(
                "INSERT INTO admins(tg_id,role,added_at) VALUES(?,?,?) ON CONFLICT(tg_id) DO NOTHING",
                (tg_id, "admin", _now()),
            )

    def list_admins(self) -> List[int]:
        with self._conn() as c:
            rows = c.execute("SELECT tg_id FROM admins").fetchall()
            return [int(r["tg_id"]) for r in rows]

    # Courses/Lessons
    def create_course(self, title: str, description: str) -> int:
        with self._conn() as c:
            cur = c.execute(
                "INSERT INTO courses(title,description,created_at) VALUES(?,?,?)",
                (title, description, _now()),
            )
            return int(cur.lastrowid)

    def list_courses(self, only_active: bool = True) -> List[sqlite3.Row]:
        with self._conn() as c:
            if only_active:
                return c.execute("SELECT * FROM courses WHERE is_active=1 ORDER BY id DESC").fetchall()
            return c.execute("SELECT * FROM courses ORDER BY id DESC").fetchall()

    def toggle_course(self, course_id: int) -> None:
        with self._conn() as c:
            c.execute("UPDATE courses SET is_active=CASE WHEN is_active=1 THEN 0 ELSE 1 END WHERE id=?", (course_id,))

    def create_lesson(self, course_id: int, title: str) -> int:
        with self._conn() as c:
            cur = c.execute(
                "INSERT INTO lessons(course_id,title,status,created_at) VALUES(?,?,?,?)",
                (course_id, title, "processing", _now()),
            )
            return int(cur.lastrowid)

    def set_lesson_status(self, lesson_id: int, status: str) -> None:
        with self._conn() as c:
            c.execute("UPDATE lessons SET status=? WHERE id=?", (status, lesson_id))

    def list_lessons(self, course_id: int) -> List[sqlite3.Row]:
        with self._conn() as c:
            return c.execute("SELECT * FROM lessons WHERE course_id=? ORDER BY id DESC", (course_id,)).fetchall()

    def add_media(self, lesson_id: int, quality: str, path: str, size_bytes: int) -> int:
        with self._conn() as c:
            cur = c.execute(
                "INSERT INTO media(lesson_id,quality,path,status,size_bytes,created_at) VALUES(?,?,?,?,?,?)",
                (lesson_id, quality, path, "ready", int(size_bytes), _now()),
            )
            return int(cur.lastrowid)

    def list_media_for_lesson(self, lesson_id: int) -> List[sqlite3.Row]:
        with self._conn() as c:
            return c.execute("SELECT * FROM media WHERE lesson_id=? AND status='ready' ORDER BY quality", (lesson_id,)).fetchall()

    def get_media(self, media_id: int) -> Optional[sqlite3.Row]:
        with self._conn() as c:
            return c.execute("SELECT * FROM media WHERE id=?", (media_id,)).fetchone()

    # Upload queue
    def create_upload(self, course_id: int, lesson_id: int, tg_file_id: str, original_path: str) -> int:
        with self._conn() as c:
            cur = c.execute(
                "INSERT INTO uploads(course_id,lesson_id,tg_file_id,original_path,status,created_at) VALUES(?,?,?,?,?,?)",
                (course_id, lesson_id, tg_file_id, original_path, "pending", _now()),
            )
            return int(cur.lastrowid)

    def claim_next_upload(self) -> Optional[sqlite3.Row]:
        with self._conn() as c:
            row = c.execute("SELECT * FROM uploads WHERE status='pending' ORDER BY id LIMIT 1").fetchone()
            if not row:
                return None
            c.execute("UPDATE uploads SET status='processing' WHERE id=?", (int(row["id"]),))
            return row

    def set_upload_status(self, upload_id: int, status: str, error: Optional[str] = None) -> None:
        with self._conn() as c:
            c.execute("UPDATE uploads SET status=?, error=? WHERE id=?", (status, error, upload_id))

    # Tokens
    def create_token(self, token: str, tg_id: int, media_id: int, expires_at: int) -> None:
        with self._conn() as c:
            c.execute(
                "INSERT INTO tokens(token,tg_id,media_id,expires_at,created_at) VALUES(?,?,?,?,?)",
                (token, tg_id, media_id, expires_at, _now()),
            )

    def get_token(self, token: str) -> Optional[sqlite3.Row]:
        with self._conn() as c:
            return c.execute("SELECT * FROM tokens WHERE token=?", (token,)).fetchone()

    def cleanup_tokens(self) -> None:
        with self._conn() as c:
            c.execute("DELETE FROM tokens WHERE expires_at < ?", (_now(),))